<?php
require_once '../../includes/config.php';

// Check if admin is logged in
if (!isAdminLoggedIn()) {
    redirect(SITE_URL . 'admin/login.php');
}

// Get order ID
$order_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($order_id <= 0) {
    redirect(SITE_URL . 'admin/orders/index.php');
}

// Handle payment verification
if (isset($_POST['verify_payment'])) {
    try {
        $conn = connectDB();
        $stmt = $conn->prepare("UPDATE orders SET payment_verified_at = NOW(), payment_notes = 'Pembayaran diverifikasi oleh admin' WHERE order_id = :order_id");
        $stmt->bindParam(':order_id', $order_id);
        $stmt->execute();
        
        $success = "Pembayaran berhasil diverifikasi";
    } catch (PDOException $e) {
        $error = "Kesalahan: " . $e->getMessage();
    }
}

// Handle payment rejection
if (isset($_POST['reject_payment'])) {
    $rejection_reason = isset($_POST['rejection_reason']) ? sanitize($_POST['rejection_reason']) : 'Bukti pembayaran tidak valid';
    
    try {
        $conn = connectDB();
        $stmt = $conn->prepare("UPDATE orders SET payment_rejected_at = NOW(), payment_notes = :rejection_reason WHERE order_id = :order_id");
        $stmt->bindParam(':rejection_reason', $rejection_reason);
        $stmt->bindParam(':order_id', $order_id);
        $stmt->execute();
        
        $success = "Pembayaran ditolak";
    } catch (PDOException $e) {
        $error = "Kesalahan: " . $e->getMessage();
    }
}

try {
    $conn = connectDB();
    
    // Get order details
    $stmt = $conn->prepare("SELECT o.*, u.name FROM orders o LEFT JOIN users u ON o.user_id = u.user_id WHERE o.order_id = :order_id");
    $stmt->bindParam(':order_id', $order_id);
    $stmt->execute();
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // If order not found or doesn't have payment proof
    if (!$order || empty($order['payment_proof'])) {
        redirect(SITE_URL . 'admin/orders/index.php');
    }
    
    $pageTitle = "Verifikasi Bukti Pembayaran #" . $order_id;
    
} catch (PDOException $e) {
    $error = "Kesalahan: " . $e->getMessage();
}

include '../header.php';
?>

<div class="admin-header">
    <h1>Verifikasi Bukti Pembayaran</h1>
    <div class="admin-actions">
        <a href="<?php echo SITE_URL; ?>admin/orders/detail.php?id=<?php echo $order_id; ?>" class="btn secondary-btn">Kembali ke Detail Pesanan</a>
    </div>
</div>

<?php if (isset($success)): ?>
    <div class="success"><?php echo $success; ?></div>
    <script>
        setTimeout(function() {
            window.location.href = '<?php echo SITE_URL; ?>admin/orders/detail.php?id=<?php echo $order_id; ?>';
        }, 3000); // Pengalihan setelah 3 detik
    </script>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="error"><?php echo $error; ?></div>
<?php endif; ?>

<div class="admin-content-box payment-proof-page">
    <div class="order-meta-info">
        <div class="order-info">
            <h3>Informasi Pesanan #<?php echo $order_id; ?></h3>
            <p><strong>Pelanggan:</strong> <?php echo $order['name']; ?></p>
            <p><strong>Total:</strong> Rp <?php echo number_format($order['total_amount'], 0, ',', '.'); ?></p>
            <p><strong>Tanggal:</strong> <?php echo date('d-m-Y H:i', strtotime($order['created_at'])); ?></p>
            <p><strong>Status:</strong> 
                <span class="status-badge status-<?php echo $order['status']; ?>">
                    <?php
                    switch($order['status']) {
                        case 'pending': echo 'Menunggu Pembayaran'; break;
                        case 'processing': echo 'Diproses'; break;
                        case 'shipped': echo 'Dikirim'; break;
                        case 'delivered': echo 'Terkirim'; break;
                        case 'cancelled': echo 'Dibatalkan'; break;
                        default: echo $order['status']; break;
                    }
                    ?>
                </span>
            </p>
        </div>
        
        <div class="verification-status">
            <?php if (isset($order['payment_verified_at'])): ?>
                <div class="status-verified">
                    <i class="fas fa-check-circle"></i> 
                    <span>Pembayaran Telah Diverifikasi</span>
                    <div class="verification-time">Diverifikasi pada <?php echo date('d-m-Y H:i', strtotime($order['payment_verified_at'])); ?></div>
                </div>
            <?php elseif (isset($order['payment_rejected_at'])): ?>
                <div class="status-rejected">
                    <i class="fas fa-times-circle"></i> 
                    <span>Pembayaran Ditolak</span>
                    <div class="verification-time">Ditolak pada <?php echo date('d-m-Y H:i', strtotime($order['payment_rejected_at'])); ?></div>
                    <?php if (!empty($order['payment_notes'])): ?>
                        <div class="rejection-reason"><?php echo htmlspecialchars($order['payment_notes']); ?></div>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="status-pending">
                    <i class="fas fa-exclamation-circle"></i> Menunggu Verifikasi
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="payment-proof-container">
        <img src="<?php echo SITE_URL . $order['payment_proof']; ?>" alt="Bukti Pembayaran" class="payment-proof-img">
    </div>
    
    <?php if (!isset($order['payment_verified_at']) && !isset($order['payment_rejected_at'])): ?>
    <div class="verification-options">
        <h3>Tindakan Verifikasi</h3>
        <div class="info-box">
            <p><i class="fas fa-info-circle"></i> <strong>Catatan:</strong> Verifikasi pembayaran terpisah dari status pesanan. Setelah memverifikasi pembayaran, Anda masih dapat mengubah status pesanan secara terpisah melalui halaman detail pesanan.</p>
        </div>
        <div class="option-buttons">
            <button id="btnVerify" class="option-btn option-verify active">
                <i class="fas fa-check-circle"></i> Verifikasi Pembayaran
            </button>
            <button id="btnReject" class="option-btn option-reject">
                <i class="fas fa-times-circle"></i> Tolak Pembayaran
            </button>
        </div>
        
        <div class="verification-forms">
            <form id="verifyForm" action="" method="post" class="verification-form active">
                <p class="form-info">Konfirmasi pembayaran ini dan ubah status pesanan menjadi "Diproses".</p>
                <div class="form-actions">
                    <button type="submit" name="verify_payment" value="1" class="btn verification-btn">
                        <i class="fas fa-check-circle"></i> Verifikasi Pembayaran
                    </button>
                </div>
            </form>
            
            <form id="rejectForm" action="" method="post" class="verification-form">
                <p class="form-info">Tolak pembayaran ini dan berikan alasan penolakan:</p>
            <div class="form-group">
                <label for="rejection_reason">Alasan Penolakan:</label>
                <textarea name="rejection_reason" id="rejection_reason" required>Bukti pembayaran tidak valid</textarea>
            </div>
                <div class="form-actions">
                    <button type="submit" name="reject_payment" value="1" class="btn rejection-btn">
                <i class="fas fa-times-circle"></i> Tolak Pembayaran
            </button>
                </div>
        </form>
        </div>
    </div>
    <?php endif; ?>
</div>

<style>
.payment-proof-page {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.order-meta-info {
    display: grid;
    grid-template-columns: 1fr auto;
    gap: 30px;
    align-items: start;
    margin-bottom: 20px;
    background-color: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.order-info h3 {
    margin-top: 0;
    color: #333;
    border-bottom: 1px solid #eee;
    padding-bottom: 10px;
    margin-bottom: 15px;
}

.status-badge {
    display: inline-block;
    padding: 3px 10px;
    border-radius: 15px;
    font-size: 0.9em;
}

.status-badge.status-pending {
    background-color: #ffc107;
    color: #212529;
}

.status-badge.status-processing {
    background-color: #17a2b8;
    color: white;
}

.status-badge.status-shipped {
    background-color: #007bff;
    color: white;
}

.status-badge.status-delivered {
    background-color: #28a745;
    color: white;
}

.status-badge.status-cancelled {
    background-color: #dc3545;
    color: white;
}

.verification-status {
    text-align: center;
    padding: 15px;
    border-radius: 5px;
    font-weight: 600;
}

.status-verified {
    color: #155724;
    background-color: #d4edda;
    border: 1px solid #c3e6cb;
    padding: 10px 15px;
    border-radius: 5px;
}

.status-rejected {
    color: #721c24;
    background-color: #f8d7da;
    border: 1px solid #f5c6cb;
    padding: 10px 15px;
    border-radius: 5px;
}

.status-pending {
    color: #856404;
    background-color: #fff3cd;
    border: 1px solid #ffeeba;
    padding: 10px 15px;
    border-radius: 5px;
}

.verification-time {
    font-size: 0.8em;
    margin-top: 5px;
    opacity: 0.8;
}

.rejection-reason {
    margin-top: 10px;
    padding: 8px;
    background-color: rgba(255,255,255,0.5);
    border-radius: 4px;
}

.payment-proof-container {
    text-align: center;
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    padding: 20px;
    margin: 20px 0;
}

.payment-proof-img {
    max-width: 100%;
    max-height: 600px;
    border: 1px solid #ddd;
    border-radius: 4px;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
}

.verification-options {
    background-color: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.verification-options h3 {
    margin-top: 0;
    color: #333;
    border-bottom: 1px solid #eee;
    padding-bottom: 10px;
    margin-bottom: 15px;
}

.info-box {
    background-color: #e8f4fd;
    border-left: 4px solid #17a2b8;
    padding: 12px 15px;
    margin-bottom: 15px;
    border-radius: 4px;
    font-size: 14px;
}

.info-box p {
    margin: 0;
}

.info-box i {
    color: #17a2b8;
    margin-right: 5px;
}

.option-buttons {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.option-btn {
    flex: 1;
    padding: 15px;
    border: 2px solid #ddd;
    border-radius: 5px;
    background-color: white;
    cursor: pointer;
    font-size: 16px;
    font-weight: 600;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    transition: all 0.3s ease;
}

.option-btn i {
    font-size: 20px;
}

.option-btn.option-verify {
    color: #555;
}

.option-btn.option-verify.active {
    border-color: #2e8540;
    background-color: #2e8540;
    color: white;
}

.option-btn.option-reject {
    color: #555;
}

.option-btn.option-reject.active {
    border-color: #2e8540;
    background-color: #2e8540;
    color: white;
}

.verification-forms {
    background-color: #f8f9fa;
    padding: 20px;
    border-radius: 5px;
    border: 1px solid #ddd;
}

.verification-form {
    display: none;
}

.verification-form.active {
    display: block;
}

.form-info {
    margin-bottom: 15px;
    color: #555;
}

.form-group {
    margin-bottom: 15px;
}

label {
    display: block;
    margin-bottom: 5px;
    font-weight: 500;
}

textarea {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    min-height: 100px;
    font-family: inherit;
}

.form-actions {
    margin-top: 20px;
}

.verification-btn, .rejection-btn {
    background-color: #2e8540;
    color: white;
    border: none;
    padding: 10px 15px;
    width: 100%;
    border-radius: 4px;
    cursor: pointer;
    font-size: 16px;
    font-weight: 600;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    transition: background-color 0.3s;
}

.verification-btn:hover, .rejection-btn:hover {
    background-color: #256a33;
}

@media (max-width: 768px) {
    .order-meta-info {
        grid-template-columns: 1fr;
    }
    
    .option-buttons {
        flex-direction: column;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const btnVerify = document.getElementById('btnVerify');
    const btnReject = document.getElementById('btnReject');
    const verifyForm = document.getElementById('verifyForm');
    const rejectForm = document.getElementById('rejectForm');
    
    if (btnVerify && btnReject && verifyForm && rejectForm) {
        btnVerify.addEventListener('click', function() {
            btnVerify.classList.add('active');
            btnReject.classList.remove('active');
            verifyForm.classList.add('active');
            rejectForm.classList.remove('active');
        });
        
        btnReject.addEventListener('click', function() {
            btnReject.classList.add('active');
            btnVerify.classList.remove('active');
            rejectForm.classList.add('active');
            verifyForm.classList.remove('active');
        });
    }
});
</script>

<?php include '../footer.php'; ?> 