<?php
require_once '../../includes/config.php';

// Check if admin is logged in
if (!isAdminLoggedIn()) {
    redirect(SITE_URL . 'admin/login.php');
}

// Get user ID
$user_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($user_id <= 0) {
    redirect(SITE_URL . 'admin/users/index.php');
}

try {
    $conn = connectDB();
    
    // Get user details
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = :user_id");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        redirect(SITE_URL . 'admin/users/index.php');
    }
    
    // Get user's orders
    $stmt = $conn->prepare("SELECT * FROM orders WHERE user_id = :user_id ORDER BY created_at DESC");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $pageTitle = "Detail Pengguna: " . $user['name'];
    
} catch (PDOException $e) {
    $error = "Kesalahan: " . $e->getMessage();
}

include '../header.php';
?>

<div class="admin-header">
    <h1>Detail Pengguna</h1>
    <div class="admin-actions">
        <a href="<?php echo SITE_URL; ?>admin/users/index.php" class="btn secondary-btn">Kembali ke Daftar Pengguna</a>
        <a href="<?php echo SITE_URL; ?>admin/users/edit.php?id=<?php echo $user_id; ?>" class="btn primary-btn">Edit Pengguna</a>
    </div>
</div>

<?php if (isset($error)): ?>
    <div class="error"><?php echo $error; ?></div>
<?php endif; ?>

<div class="admin-content-box">
    <div class="user-details">
        <div class="user-profile">
            <div class="profile-section">
                <h3>Informasi Akun</h3>
                <div class="detail-item">
                    <strong>ID Pengguna:</strong> <?php echo $user['user_id']; ?>
                </div>
                <div class="detail-item">
                    <strong>Nama:</strong> <?php echo $user['name']; ?>
                </div>
                <div class="detail-item">
                    <strong>Username:</strong> <?php echo $user['username']; ?>
                </div>
                <div class="detail-item">
                    <strong>Email:</strong> <?php echo $user['email']; ?>
                </div>
                <div class="detail-item">
                    <strong>Telepon:</strong> <?php echo $user['phone'] ?? 'Tidak ada'; ?>
                </div>
                <div class="detail-item">
                    <strong>Tanggal Daftar:</strong> <?php echo date('d-m-Y H:i', strtotime($user['created_at'])); ?>
                </div>
            </div>
            
            <div class="profile-section">
                <h3>Alamat</h3>
                <div class="detail-item address">
                    <?php echo !empty($user['address']) ? nl2br($user['address']) : 'Belum ada alamat'; ?>
                </div>
            </div>
        </div>
        
        <div class="user-activity">
            <h3>Riwayat Pesanan</h3>
            <?php if (empty($orders)): ?>
                <p class="no-data">Belum ada pesanan</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="admin-table">
                        <thead>
                            <tr>
                                <th>ID Pesanan</th>
                                <th>Tanggal</th>
                                <th>Total</th>
                                <th>Status</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                                <tr>
                                    <td>#<?php echo $order['order_id']; ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($order['created_at'])); ?></td>
                                    <td>Rp <?php echo number_format($order['total_amount'], 0, ',', '.'); ?></td>
                                    <td>
                                        <span class="status-badge status-<?php echo $order['status']; ?>">
                                            <?php
                                            switch($order['status']) {
                                                case 'pending': echo 'Menunggu Pembayaran'; break;
                                                case 'processing': echo 'Diproses'; break;
                                                case 'shipped': echo 'Dikirim'; break;
                                                case 'delivered': echo 'Terkirim'; break;
                                                case 'cancelled': echo 'Dibatalkan'; break;
                                                default: echo $order['status']; break;
                                            }
                                            ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="<?php echo SITE_URL; ?>admin/orders/detail.php?id=<?php echo $order['order_id']; ?>" class="action-btn view-btn" title="Detail">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
    .user-details {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }
    
    .profile-section {
        background-color: #f9f9f9;
        border-radius: 8px;
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .profile-section h3 {
        margin-top: 0;
        margin-bottom: 15px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e0e0e0;
        color: #333;
    }
    
    .detail-item {
        margin-bottom: 10px;
        padding-bottom: 10px;
        border-bottom: 1px solid #ececec;
    }
    
    .detail-item:last-child {
        margin-bottom: 0;
        padding-bottom: 0;
        border-bottom: none;
    }
    
    .detail-item strong {
        display: inline-block;
        width: 120px;
    }
    
    .detail-item.address {
        white-space: pre-line;
    }
    
    .no-data {
        color: #666;
        font-style: italic;
    }
    
    @media (max-width: 768px) {
        .user-details {
            grid-template-columns: 1fr;
        }
    }
</style>

<?php include '../footer.php'; ?> 