<?php
require_once '../includes/config.php';

// Check if user is already logged in
if (isUserLoggedIn()) {
    redirect(SITE_URL);
}

$error = '';
$success = false;
$name = '';
$username = '';
$email = '';
$address = '';
$phone = '';

// Process registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = sanitize($_POST['name']);
    $username = sanitize($_POST['username']);
    $email = sanitize($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $address = sanitize($_POST['address']);
    $phone = sanitize($_POST['phone']);
    
    // Validate input
    if (empty($name) || empty($username) || empty($email) || empty($password) || empty($confirm_password)) {
        $error = 'Silakan lengkapi semua field yang wajib diisi';
    } elseif ($password !== $confirm_password) {
        $error = 'Password dan konfirmasi password tidak cocok';
    } elseif (strlen($password) < 6) {
        $error = 'Password harus minimal 6 karakter';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Format email tidak valid';
    } else {
        try {
            $conn = connectDB();
            
            // Check if username already exists
            $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username");
            $stmt->bindParam(':username', $username);
            $stmt->execute();
            
            if ($stmt->fetch(PDO::FETCH_ASSOC)) {
                $error = 'Username sudah digunakan. Silakan pilih username lain';
            } else {
                // Check if email already exists
                $stmt = $conn->prepare("SELECT * FROM users WHERE email = :email");
                $stmt->bindParam(':email', $email);
                $stmt->execute();
                
                if ($stmt->fetch(PDO::FETCH_ASSOC)) {
                    $error = 'Email sudah terdaftar. Silakan gunakan email lain';
                } else {
                    // Hash the password
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    
                    // Insert new user
                    $stmt = $conn->prepare("INSERT INTO users (name, username, email, password, address, phone) VALUES (:name, :username, :email, :password, :address, :phone)");
                    $stmt->bindParam(':name', $name);
                    $stmt->bindParam(':username', $username);
                    $stmt->bindParam(':email', $email);
                    $stmt->bindParam(':password', $hashedPassword);
                    $stmt->bindParam(':address', $address);
                    $stmt->bindParam(':phone', $phone);
                    
                    if ($stmt->execute()) {
                        $success = true;
                        $name = '';
                        $username = '';
                        $email = '';
                        $address = '';
                        $phone = '';
                    } else {
                        $error = 'Terjadi kesalahan saat mendaftarkan akun';
                    }
                }
            }
        } catch (PDOException $e) {
            $error = 'Terjadi kesalahan: ' . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registrasi - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="<?php echo SITE_URL; ?>assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background-color: #f4f4f4;
            margin: 0;
            font-family: Arial, sans-serif;
            padding: 40px 0;
        }
        
        .user-register-container {
            background: white;
            padding: 30px;
            border-radius: 5px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            width: 100%;
            max-width: 600px;
            margin: 0 auto;
        }
        
        .user-register-container h2 {
            text-align: center;
            color: #333;
            margin-bottom: 20px;
        }
        
        .user-register-logo {
            text-align: center;
            margin-bottom: 20px;
        }
        
        .user-register-logo h1 {
            color: var(--primary-color);
            font-size: 24px;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-sizing: border-box;
        }
        
        .form-group textarea {
            height: 100px;
            resize: vertical;
        }
        
        .form-group small {
            display: block;
            margin-top: 5px;
            color: #666;
            font-size: 12px;
        }
        
        .form-group .required {
            color: #e74c3c;
        }
        
        .btn {
            display: block;
            width: 100%;
            padding: 12px;
            background-color: var(--primary-color);
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 16px;
            cursor: pointer;
            text-align: center;
            margin-top: 20px;
        }
        
        .btn:hover {
            background-color: #3a8f3a;
        }
        
        .error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 12px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .success {
            background-color: #d4edda;
            color: #155724;
            padding: 12px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .success a {
            color: #155724;
            font-weight: bold;
            text-decoration: underline;
        }
        
        .user-register-footer {
            margin-top: 20px;
            text-align: center;
            font-size: 14px;
        }
        
        .user-register-footer a {
            color: var(--primary-color);
            text-decoration: none;
        }
        
        .user-register-footer a:hover {
            text-decoration: underline;
        }
        
        .back-to-site {
            display: block;
            text-align: center;
            margin-top: 15px;
            color: #666;
        }
        
        /* Password visibility toggle */
        .password-input-wrapper {
            position: relative;
        }
        
        .password-toggle {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #666;
            z-index: 10;
        }
        
        .password-toggle:hover {
            color: var(--primary-color);
        }
        
        /* Media queries for responsiveness */
        @media screen and (max-width: 640px) {
            .user-register-container {
                padding: 20px;
                margin: 0 15px;
            }
            .user-register-logo h1 { font-size: 22px; }
            .user-register-container h2 { font-size: 24px; }
            body { padding: 20px 0; }
        }
    </style>
</head>
<body>
    <div class="user-register-container">
        <div class="user-register-logo">
            <h1><?php echo SITE_NAME; ?></h1>
        </div>
        
        <h2>Daftar Akun</h2>
        
        <?php if ($error): ?>
            <div class="error"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success">
                Pendaftaran berhasil! Silakan <a href="login.php">login</a> dengan akun baru Anda.
            </div>
        <?php else: ?>
            <form action="register.php" method="post">
                <div class="form-group">
                    <label for="name">Nama Lengkap: <span class="required">*</span></label>
                    <input type="text" id="name" name="name" value="<?php echo $name; ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="username">Username: <span class="required">*</span></label>
                    <input type="text" id="username" name="username" value="<?php echo $username; ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="email">Email: <span class="required">*</span></label>
                    <input type="email" id="email" name="email" value="<?php echo $email; ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="password">Password: <span class="required">*</span></label>
                    <div class="password-input-wrapper">
                        <input type="password" id="password" name="password" required>
                        <span class="password-toggle" onclick="togglePassword('password')">
                            <i class="far fa-eye" id="password-toggle-icon"></i>
                        </span>
                    </div>
                    <small>Password minimal 6 karakter</small>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Konfirmasi Password: <span class="required">*</span></label>
                    <div class="password-input-wrapper">
                        <input type="password" id="confirm_password" name="confirm_password" required>
                        <span class="password-toggle" onclick="togglePassword('confirm_password')">
                            <i class="far fa-eye" id="confirm_password-toggle-icon"></i>
                        </span>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="address">Alamat:</label>
                    <textarea id="address" name="address"><?php echo $address; ?></textarea>
                </div>
                
                <div class="form-group">
                    <label for="phone">Nomor Telepon:</label>
                    <input type="text" id="phone" name="phone" value="<?php echo $phone; ?>">
                </div>
                
                <button type="submit" class="btn">Daftar</button>
            </form>
            
            <div class="user-register-footer">
                <p>Sudah memiliki akun? <a href="login.php">Login di sini</a></p>
                <a href="<?php echo SITE_URL; ?>" class="back-to-site">
                    <i class="fas fa-arrow-left"></i> Kembali ke Website
                </a>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        function togglePassword(inputId) {
            const passwordInput = document.getElementById(inputId);
            const toggleIcon = document.getElementById(inputId + '-toggle-icon');
            
            if (passwordInput) {
                if (passwordInput.type === 'password') {
                    passwordInput.type = 'text';
                    toggleIcon.classList.remove('fa-eye');
                    toggleIcon.classList.add('fa-eye-slash');
                } else {
                    passwordInput.type = 'password';
                    toggleIcon.classList.remove('fa-eye-slash');
                    toggleIcon.classList.add('fa-eye');
                }
            }
        }
    </script>
</body>
</html> 